<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Admin; // The only model needed for this controller
use App\Models\User; // Import the User model
use App\Notifications\SendPushNotification; // Import the SendPushNotification
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

class AdminController extends Controller
{
    /**
     * Send an invisible push notification to a user's device to trigger session revocation.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Http\JsonResponse
     */
    public function sendRevokeSessionNotification(User $user)
    {
        // Send a data-only push notification (invisible to the user)
        $user->notify(new SendPushNotification(
            null, // No title
            null, // No body
            'session_revoked', // Custom type to signal session revocation
            ['action' => 'logout'] // Additional data for the client app
        ));

        return response()->json(['message' => 'Session revocation notification sent successfully to user ' . $user->id]);
    }

    /**
     * Display a listing of the admins.
     */
    public function index(Request $request)
    {
        $query = Admin::select('id', 'username');

        if ($request->filled('search')) {
            $searchTerm = $request->input('search');
            $query->where('username', 'like', "%{$searchTerm}%");
        }

        $admins = $query->get();
        return response()->json($admins);
    }

    /**
     * Store a newly created admin in storage.
     */
    public function store(Request $request)
    {
        try {
            // Use the validated data directly.
            $validatedData = $request->validate([
                'username' => 'required|string|unique:admins,username',
                'password' => 'required|string|min:8',
            ]);

            $admin = Admin::create([
                'username' => $validatedData['username'],
                'password' => Hash::make($validatedData['password']),
            ]);

            return response()->json(['message' => "Admin user {$admin->username} created successfully"], 201);
        } catch (ValidationException $e) {
            // It's good practice to return the standard Laravel validation response.
            return response()->json(['errors' => $e->errors()], 422);
        }
    }

    /**
     * Display the specified admin.
     */
    public function show($id)
    {
        // Your existing code is perfect. findOrFail is the correct choice.
        $admin = Admin::select('id', 'username')->findOrFail($id);
        return response()->json($admin);
    }

    /**
     * Update the specified admin in storage.
     */
    public function update(Request $request, $id)
    {
        try {
            $admin = Admin::findOrFail($id);

            $validatedData = $request->validate([
                'username' => 'sometimes|required|string|unique:admins,username,' . $admin->id,
                'password' => 'nullable|string|min:8',
            ]);

            // Update the admin with the validated data.
            if (isset($validatedData['username'])) {
                $admin->username = $validatedData['username'];
            }


            // Only hash and update the password if a new one was provided.
            if (!empty($validatedData['password'])) {
                $admin->password = Hash::make($validatedData['password']);
            }
            $admin->save();

            return response()->json(['message' => "Admin user {$admin->username} updated successfully"]);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        }
    }

    /**
     * Remove the specified admin from storage.
     */
    public function destroy($id)
    {
        try {
            $admin = Admin::findOrFail($id);

            // Added a check to prevent deleting the last admin, which is a good safety measure.
            if (Admin::count() === 1) {
                return response()->json(['error' => 'Cannot delete the last admin user.'], 403);
            }

            $admin->delete();

            return response()->json(['message' => "Admin user with ID {$id} deleted successfully"]);
        } catch (\Exception $e) {
            // Providing the actual error message can be useful for debugging in non-production environments.
            return response()->json(['error' => 'Failed to delete admin user.', 'message' => $e->getMessage()], 500);
        }
    }

    /**
     * Reset the password for a specific admin.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function resetPassword(Request $request, $id)
    {
        try {
            $admin = Admin::findOrFail($id);

            $validatedData = $request->validate([
                'password' => 'required|string|min:8',
            ]);

            $admin->password = Hash::make($validatedData['password']);
            $admin->save();

            return response()->json(['message' => "Password for admin user {$admin->username} has been reset successfully."]);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Failed to reset password.', 'message' => $e->getMessage()], 500);
        }
    }
}