<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Advertisement;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use League\Flysystem\UnableToRetrieveMetadata;

class AdvertisementController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        $query = Advertisement::query();

        // Search functionality
        if ($request->filled('search')) {
            $searchTerm = $request->input('search');
            $query->where(function ($q) use ($searchTerm) {
                $q->where('title', 'like', "%{$searchTerm}%")
                  ->orWhere('description', 'like', "%{$searchTerm}%")
                  ->orWhere('target_url', 'like', "%{$searchTerm}%");
            });
        }

        // Admin-specific logic
        if ($request->user() && $request->user()->isAdmin()) {
            if (!$request->query('all')) {
                 $query->where('is_active', true);
            }
        } else {
            // Non-admins only see active ads
            $query->where('is_active', true);
        }

        $advertisements = $query->latest()->get();

        return response()->json(['data' => $advertisements]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'target_url' => 'nullable|url',
            'image' => 'required|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $validated = $validator->validated();
        $validated['image_url'] = $request->file('image')->store('advertisements', 'public');

        $advertisement = Advertisement::create($validated);

        return response()->json($advertisement, 201);
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Advertisement  $advertisement
     * @return \Illuminate\Http\JsonResponse
     */
    public function show(Advertisement $advertisement)
    {
        return response()->json($advertisement);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Advertisement  $advertisement
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, Advertisement $advertisement)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'sometimes|required|string|max:255',
            'description' => 'nullable|string',
            'target_url' => 'nullable|url', // Corrected validation
            'image' => 'sometimes|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'is_active' => 'sometimes|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $data = $validator->validated();

        if ($request->hasFile('image')) {
            // Delete old image
            if ($advertisement->image_url) {
                Storage::disk('public')->delete($advertisement->image_url);
            }
            $path = $request->file('image')->store('advertisements', 'public');
            $data['image_url'] = $path;
        }

        $advertisement->update($data);

        return response()->json($advertisement);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Advertisement  $advertisement
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy(Advertisement $advertisement)
    {
        // Delete the image file
        if ($advertisement->image_url) {
            Storage::disk('public')->delete($advertisement->image_url);
        }
        
        $advertisement->delete();

        return response()->json(null, 204);
    }

    /**
     * Toggle the active status of the specified resource.
     *
     * @param  \App\Models\Advertisement  $advertisement
     * @return \Illuminate\Http\JsonResponse
     */
    public function toggleActive(Advertisement $advertisement)
    {
        $advertisement->update(['is_active' => !$advertisement->is_active]);

        return response()->json($advertisement);
    }
}