<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\AuthCode;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash; // Make sure Hash is imported
use Illuminate\Validation\ValidationException;
use App\Models\Admin; // Make sure Admin is imported

class AuthController extends Controller
{
    /**
     * Handle a login request for an Admin user.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function login(Request $request)
    {
        $credentials = $request->validate([
            'username' => 'required|string',
            'password' => 'required|string',
        ]);

        // 1. Find the admin by the provided username.
        $admin = Admin::where('username', $credentials['username'])->first();

        // 2. Securely check if the admin exists and if the provided password matches the hashed password in the database.
        //    This is the correct, secure way to verify credentials for a stateless API login.
        if (!$admin || !Hash::check($credentials['password'], $admin->password)) {
            throw ValidationException::withMessages([
                'username' => ['The provided credentials do not match our records.'],
            ]);
        }

        // 3. If the credentials are valid, create and return the Sanctum token.
        $token = $admin->createToken('admin-token', ['admin'])->plainTextToken;

        return response()->json([
            'access_token' => $token,
            'token_type' => 'Bearer',
            'user' => [
                'id' => $admin->id,
                'name' => $admin->username, // Assuming 'username' is the display name for admin
                'role' => 'admin',
            ],
        ]);
    }

    /**
     * Validate an authentication code and issue a Sanctum token for a regular User.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function validateCode(Request $request)
    {
        // Use Laravel's built-in `validate` method. It's cleaner and throws
        // a formatted JSON error response on failure automatically.
        $validated = $request->validate([
            'authentication_code' => 'required|string|min:6|exists:auth_codes,code',
        ]);

        // Eager load the 'user' relationship to prevent an extra query.
        $authCode = AuthCode::with('user')->where('code', $validated['authentication_code'])->first();

        // Combine all validation checks for better clarity.
        if ($authCode->is_used || ($authCode->expires_at && $authCode->expires_at->isPast())) {
            return response()->json(['error' => 'Invalid or expired authentication code.'], 401);
        }

        $user = $authCode->user;
        if (!$user) {
            return response()->json(['error' => 'User associated with this code not found.'], 404);
        }

        // Mark the code as used to prevent replay attacks.
        $authCode->update(['is_used' => true]);

        // Create a token for the user with 'user' ability.
        $token = $user->createToken('auth-token', ['user'])->plainTextToken;

        return response()->json([
            'access_token' => $token,
            'token_type' => 'Bearer',
            'user' => [
                'id' => $user->id,
                'email' => $user->email,
                'name' => $user->name,
            ],
        ]);
    }

    /**
     * Log out the currently authenticated user or admin.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function logout(Request $request)
    {
        // The if/else check is not necessary. Sanctum correctly identifies the
        // authenticated model (whether User or Admin) via the token.
        // This single line works for both cases.
        $request->user()->currentAccessToken()->delete();

        return response()->json(['message' => 'Logged out successfully']);
    }
}