<?php

namespace App\Http\Controllers\Api;

use App\Events\InvoiceCreated;
use App\Events\InvoiceStatusUpdated;
use App\Http\Controllers\Controller;
use App\Http\Resources\InvoiceResource;
use App\Models\Invoice;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class InvoiceController extends Controller
{
    /**
     * Display a paginated listing of invoices.
     */
    public function index(Request $request)
    {
        $query = Invoice::with(['user', 'project']);
        $user = $request->user();

        if ($user->tokenCan('admin')) {
            // Admin: Can view all invoices and filter by user
            if ($request->has('user_id')) {
                $query->where('user_id', $request->input('user_id'));
            }
        } else {
            // Regular User: Can only view their own invoices
            $query->where('user_id', $user->id);
        }

        $invoices = $query->latest()->paginate(15);

        return InvoiceResource::collection($invoices);
    }

    /**
     * Store a newly created invoice and its file in storage.
     * (Admin Only)
     */
    public function store(Request $request)
    {
        // AUTHORIZATION: Only admins can create invoices.
        $this->authorize('create', Invoice::class);

        $validatedData = $request->validate([
            'user_id' => 'required|exists:users,id',
            'project_id' => 'nullable|exists:projects,id',
            'invoice_number' => 'required|string|max:255|unique:invoices',
            'amount' => 'required|numeric|min:0',
            'due_date' => 'required|date',
            'status' => ['required', Rule::in(['Due', 'Paid', 'Pending', 'Overdue'])],
            'file' => 'required|file|mimes:pdf|max:10240',
        ]);

        $path = null;
        try {
            // SECURE: Server determines the user_id and file path.
            $userIdForPath = $validatedData['user_id'];
            $directory = "uploads/{$userIdForPath}/invoices";
            
            // The `store` method will automatically generate a unique filename.
            $path = $request->file('file')->store($directory, 'public');

            if (!$path) {
                throw new \Exception('File could not be stored.');
            }
            
            // Add the file path to the data for creation.
            $validatedData['file_url'] = $path;

            $invoice = Invoice::create($validatedData);

            event(new InvoiceCreated($invoice));

            return new InvoiceResource($invoice);

        } catch (\Throwable $e) {
            // If anything fails, delete the uploaded file to prevent orphaned files.
            if ($path) {
                Storage::disk('public')->delete($path);
            }
            \Log::error('Invoice creation failed: ' . $e->getMessage());
            return response()->json(['error' => 'Invoice creation failed.'], 500);
        }
    }

    /**
     * Display the specified invoice.
     */
    public function show(Invoice $invoice)
    {
        $this->authorize('view', $invoice);
        return new InvoiceResource($invoice->load(['user', 'project']));
    }

    /**
     * Update the specified invoice in storage.
     * (Admin Only)
     */
    public function update(Request $request, Invoice $invoice)
    {
        $this->authorize('update', $invoice);

        $validatedData = $request->validate([
            'invoice_number' => ['sometimes', 'required', 'string', 'max:255', Rule::unique('invoices')->ignore($invoice->id)],
            'amount' => 'sometimes|required|numeric|min:0',
            'due_date' => 'sometimes|required|date',
            'status' => ['sometimes', 'required', Rule::in(['Due', 'Paid', 'Pending', 'Overdue'])],
        ]);

        $invoice->update($validatedData);

        event(new InvoiceStatusUpdated($invoice));

        return new InvoiceResource($invoice);
    }

    /**
     * Remove the specified invoice from storage.
     */
    public function destroy(Invoice $invoice)
    {
        $this->authorize('delete', $invoice);

        // Use a transaction to ensure both file and record are deleted.
        DB::transaction(function () use ($invoice) {
            if ($invoice->file_url) {
                Storage::disk('public')->delete($invoice->file_url);
            }
            $invoice->delete();
        });

        return response()->json(null, 204);
    }

    /**
     * Download the file associated with a specific invoice.
     *
     * @param  \App\Models\Invoice  $invoice
     * @return \Symfony\Component\HttpFoundation\StreamedResponse
     */
    public function download(Invoice $invoice)
    {
        // AUTHORIZATION: Use the policy to ensure the user owns this invoice.
        // This is the crucial security step that the old endpoint was missing.
        $this->authorize('view', $invoice);

        // Check if the file exists on the server.
        if (!$invoice->file_url || !Storage::disk('public')->exists($invoice->file_url)) {
            abort(404, 'File not found.');
        }

        // Return the file as a download response.
        return Storage::disk('public')->download($invoice->file_url);
    }
}
