<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Pagination\LengthAwarePaginator;
use SplFileObject; // Use the Standard PHP Library (SPL) for efficient file reading

class LogController extends Controller
{
    /**
     * The maximum number of log lines to read from the file.
     * This is a safety valve to prevent memory exhaustion on huge log files.
     * @var int
     */
    protected const MAX_LINES_TO_READ = 10000;

    /**
     * Display a paginated list of the most recent log entries.
     * (Admin Only)
     */
    public function index(Request $request)
    {
        $validated = $request->validate([
            'page' => 'nullable|integer|min:1',
            'per_page' => 'nullable|integer|min:1|max:100',
            'all' => 'nullable|boolean', // <-- Add validation for the new parameter
        ]);

        $logPath = storage_path('logs/laravel.log');
        if (!file_exists($logPath)) {
            return response()->json(['data' => []]);
        }

        try {
            $file = new \SplFileObject($logPath, 'r');
            $file->setFlags(\SplFileObject::SKIP_EMPTY);
            $file->seek(PHP_INT_MAX);
            $lastLine = $file->key();
            $startLine = max(0, $lastLine - self::MAX_LINES_TO_READ);
            
            $lines = [];
            $file->seek($startLine);
            while (!$file->eof() && $file->key() <= $lastLine) {
                $lines[] = rtrim($file->current());
                $file->next();
            }
            $lines = array_reverse($lines); // Newest logs first

            // --- CONDITIONAL RESPONSE ---
            if (isset($validated['all']) && $validated['all']) {
                // If 'all=true' is requested, return the entire (but capped) log array.
                return response()->json(['data' => $lines]);
            }

            // --- DEFAULT PAGINATED RESPONSE ---
            $perPage = $validated['per_page'] ?? 15;
            $currentPage = $validated['page'] ?? 1;
            $paginator = new \Illuminate\Pagination\LengthAwarePaginator(
                array_slice($lines, ($currentPage - 1) * $perPage, $perPage),
                count($lines), // Total items is the count of our capped array
                $perPage,
                $currentPage,
                ['path' => $request->url()]
            );
            return response()->json($paginator);

        } catch (\Exception $e) {
            \Log::error('Could not read log file: ' . $e->getMessage());
            return response()->json(['error' => 'Could not read or process log file.'], 500);
        }
    }
}