<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Resources\NotificationResource;
use App\Models\Notification;
use Illuminate\Http\Request;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log; // Import the Log facade

class NotificationController extends Controller
{
    /**
     * Display a paginated list of the user's notifications.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        $userId = $user->id;

        // Get user-specific notifications with pivot data
        $userNotifications = $user->notifications()
            ->wherePivotNull('deleted_at')
            ->withPivot(['read_at', 'deleted_at', 'created_at', 'updated_at'])
            ->get()
            ->map(function ($notification) {
                // Transform to have consistent structure
                return [
                    'id' => $notification->id,
                    'title' => $notification->title,
                    'body' => $notification->body,
                    'type' => $notification->type,
                    'created_at' => $notification->created_at,
                    'updated_at' => $notification->updated_at,
                    'read_at' => $notification->pivot->read_at,
                    'deleted_at' => $notification->pivot->deleted_at,
                ];
            });

        // Get broadcast notifications that the user has not soft-deleted
        $broadcastNotifications = Notification::where('type', 'broadcast')
            ->whereDoesntHave('users', function ($query) use ($user) {
                $query->where('user_id', $user->id)
                      ->whereNotNull('notification_user.deleted_at');
            })
            ->get()
            ->map(function ($notification) use ($user) {
                // For the remaining notifications, get their read status
                $pivotModel = $user->notifications()->where('notification_id', $notification->id)->first();
                
                return [
                    'id' => $notification->id,
                    'title' => $notification->title,
                    'body' => $notification->body,
                    'type' => $notification->type,
                    'created_at' => $notification->created_at,
                    'updated_at' => $notification->updated_at,
                    'read_at' => $pivotModel ? $pivotModel->pivot->read_at : null,
                    'deleted_at' => null, // This is null because we already filtered them by query
                ];
            });

        // Merge and sort all notifications
        $allNotifications = $userNotifications
            ->concat($broadcastNotifications)
            ->sortByDesc('created_at')
            ->values(); // Reset array keys

        // Manual pagination
        $perPage = 15;
        $currentPage = $request->input('page', 1);
        $currentPageItems = $allNotifications->slice(($currentPage - 1) * $perPage, $perPage)->values();

        $paginatedResult = new LengthAwarePaginator(
            $currentPageItems,
            count($allNotifications),
            $perPage,
            $currentPage,
            [
                'path' => $request->url(), 
                'query' => $request->query()
            ]
        );

        return response()->json([
            'data' => $currentPageItems,
            'current_page' => $paginatedResult->currentPage(),
            'last_page' => $paginatedResult->lastPage(),
            'per_page' => $paginatedResult->perPage(),
            'total' => $paginatedResult->total(),
        ]);
    }

    /**
     * Store a new notification and attach it to users.
     * (Admin Only)
     */
    public function store(Request $request)
    {
        $this->authorize('create', Notification::class);

        $validatedData = $request->validate([
            'title' => 'required|string|max:255',
            'body' => 'required|string',
            'type' => 'required|in:user,broadcast',
            'category' => 'nullable|string|max:255', // Make category nullable
            'user_ids' => 'required_if:type,user|array',
            'user_ids.*' => 'exists:users,id',
        ]);

        $notification = Notification::create($validatedData);

        if ($validatedData['type'] === 'user' && !empty($validatedData['user_ids'])) {
            $notification->users()->attach($validatedData['user_ids']);
            Log::info('NotificationController: Users attached to notification.', [
                'notification_id' => $notification->id,
                'attached_user_ids' => $validatedData['user_ids'],
                'current_attached_users_count' => $notification->users()->count(), // Check count after attach
            ]);
            $notification->load('users'); // Reload the notification with its attached users
            \App\Services\NotificationService::sendPushNotifications($notification); // Send push notifications
        } elseif ($validatedData['type'] === 'broadcast') {
            // For broadcast notifications, we send to all users who have push tokens
            \App\Services\NotificationService::sendPushNotifications($notification);
        }

        // The NotificationObserver is no longer used for sending push notifications.

        return new NotificationResource($notification);
    }

    /**
     * Mark a notification as read for the authenticated user.
     * This works for both user-specific and broadcast notifications.
     *
     * @param  \App\Models\Notification  $notification
     * @return \Illuminate\Http\JsonResponse
     */
    public function markAsRead(Notification $notification)
    {
        $user = Auth::user();

        // Use updateOrCreate to find the pivot record or create one on the fly,
        // then update the read_at timestamp.
        $pivot = $user->notifications()->where('notification_id', $notification->id)->first();

        if ($pivot) {
            $user->notifications()->updateExistingPivot($notification->id, [
                'read_at' => now(),
            ]);
        } else {
            $user->notifications()->attach($notification->id, [
                'read_at' => now(),
            ]);
        }

        return response()->json(['message' => 'Notification marked as read.']);
    }

    /**
     * Mark a notification as deleted for the authenticated user (soft delete).
     * This works for both user-specific and broadcast notifications.
     *
     * @param  \App\Models\Notification  $notification
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy(Notification $notification)
    {
        $user = Auth::user();

        // Find the pivot record or create it, then set deleted_at
        $pivot = $user->notifications()->where('notification_id', $notification->id)->first();

        if ($pivot) {
            $user->notifications()->updateExistingPivot($notification->id, [
                'deleted_at' => now(),
            ]);
        } else {
            $user->notifications()->attach($notification->id, [
                'deleted_at' => now(),
            ]);
        }

        return response()->json([
            'message' => 'Notification has been removed.',
            'deleted_at' => now()->toISOString(), // Return the deletion timestamp
        ]);
    }

    /**
     * Get the count of unread notifications for the authenticated user.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function unreadCount()
    {
        $user = Auth::user();
        $userId = $user->id;

        // 1. Count unread user-specific notifications
        $unreadUserNotifications = $user->notifications()
            ->whereNull('notification_user.read_at')
            ->whereNull('notification_user.deleted_at')
            ->count();

        // 2. Count unread broadcast notifications
        // This counts all broadcasts and subtracts the ones the user has interacted with
        $unreadBroadcastNotifications = Notification::where('type', 'broadcast')
            ->whereDoesntHave('users', function ($query) use ($userId) {
                // Exclude broadcasts where a pivot record exists for this user
                $query->where('user_id', $userId);
            })
            ->count();
        
        // 3. Sum them up for the total
        $totalUnreadCount = $unreadUserNotifications + $unreadBroadcastNotifications;

        return response()->json(['unread_count' => $totalUnreadCount]);
    }

    /**
     * Restore a soft-deleted notification for the authenticated user.
     * (Optional - if you want to allow undoing deletions)
     *
     * @param  \App\Models\Notification  $notification
     * @return \Illuminate\Http\JsonResponse
     */
    public function restore(Notification $notification)
    {
        $user = Auth::user();

        // Find the pivot record and remove the deleted_at timestamp
        $pivot = $user->notifications()->where('notification_id', $notification->id)->first();

        if ($pivot) {
            $user->notifications()->updateExistingPivot($notification->id, [
                'deleted_at' => null,
            ]);
        } else {
            // If there's no pivot, it means it was never "deleted" by this user,
            // so we attach it without a deleted_at timestamp.
            $user->notifications()->attach($notification->id, [
                'deleted_at' => null,
            ]);
        }

        return response()->json(['message' => 'Notification restored.']);
    }
}