<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Resources\ProfileResource; // <-- Import the new resource
use Illuminate\Http\Request;

class ProfileController extends Controller
{
    /**
     * Display the authenticated user's profile.
     * Corresponds to the GET /my-profile route.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \App\Http\Resources\ProfileResource
     */
    public function show(Request $request)
    {
        // Get the authenticated user and eager load their profile.
        $user = $request->user()->load('profile');

        // If a user has no profile yet, this will gracefully handle it.
        // We can create one on-the-fly to ensure the frontend always gets an object.
        $profile = $user->profile ?? new \App\Models\Profile(['full_name' => $user->name]);

        return new ProfileResource($profile);
    }

    /**
     * Update the authenticated user's profile.
     * Corresponds to the PUT /my-profile route.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request)
    {
        // Validate the incoming data against our desired fields.
        $validatedData = $request->validate([
            'full_name' => 'sometimes|required|string|max:255',
            'phone_number' => 'nullable|string|max:255',
            'company' => 'nullable|string|max:255',
            'ICE' => 'nullable|string|max:255',
            'avatar_url' => 'nullable|string|url',
            'locale' => 'nullable|string|in:en,fr',
        ]);

        // Get the authenticated user.
        $user = $request->user();

        // Use updateOrCreate. This is robust: it will update the user's
        // existing profile or create one if it doesn't exist.
        $profile = $user->profile()->updateOrCreate(
            ['user_id' => $user->id], // Condition to find the profile
            $validatedData // Data to update or create with
        );

        return response()->json([
            'message' => 'Profile updated successfully.',
            'profile' => new ProfileResource($profile->load('user')), // Return the updated data
        ]);
    }
}
