<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Resources\ProjectResource;
use App\Models\Project;
use App\Jobs\SendNewProjectNotification;
use App\Events\ProjectStatusUpdated;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log; // Import the Log facade
use Illuminate\Support\Facades\Mail;
use App\Mail\EventTriggeredMail;
use App\Models\Admin;
use Illuminate\Support\Facades\DB;

class ProjectController extends Controller
{

    /**
     * Display a paginated listing of projects.
     * - Regular users see only their own projects.
     * - Admins see all projects and can filter by user_id.
     */
    public function index(Request $request)
    {
        $query = Project::with('service');
        $user = $request->user();

        if ($user->tokenCan('admin')) {
            // Admin: Can view all projects and filter by user
            if ($request->has('user_id')) {
                $query->where('user_id', $request->input('user_id'));
            }
        } else {
            // Regular User: Can only view their own projects
            $query->where('user_id', $user->id);
        }

        $projects = $query->latest()->paginate(15);

        return ProjectResource::collection($projects);
    }

    /**
     * Store a newly created project in storage.
     */
    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'service_id' => 'required|exists:services,id',
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'details' => 'nullable|array',
            'status' => 'required|string|max:255',
            'file' => 'nullable|file|max:102400', // 100MB max
        ]);

        $project = null;
        $path = null;

        try {
            DB::transaction(function () use ($request, $validatedData, &$project, &$path) {
                // Create the project first
                $project = $request->user()->projects()->create([
                    'service_id' => $validatedData['service_id'],
                    'name' => $validatedData['name'],
                    'description' => $validatedData['description'] ?? null,
                    'details' => $validatedData['details'] ?? null,
                    'status' => $validatedData['status'],
                ]);

                // If a file is included, store it and update the project
                if ($request->hasFile('file')) {
                    $directory = "uploads/{$project->user_id}/projects/{$project->id}";
                    $path = $request->file('file')->store($directory, 'public');

                    $fileMetadata = [
                        'path' => $path,
                        'original_name' => $request->file('file')->getClientOriginalName(),
                        'uploaded_at' => now()->toDateTimeString(),
                        'url' => Storage::url($path),
                        'file_type' => $request->file('file')->getClientMimeType(),
                    ];

                    // Add file metadata to the project's details
                    $details = $project->details ?? [];
                    $details['files'][] = $fileMetadata;
                    $project->details = $details;
                    $project->save();
                }

                // Get owner information
                $ownerName = $request->user()->name;
                $ownerOffice = $request->user()->profile->company ?? 'N/A'; // Assuming profile has a company field

                // Get service name
                $serviceName = $project->service->name ?? 'N/A';

                // Send email to admin, only if the project is not a draft
                if ($project->status !== 'draft') {
                    $subject = 'Nouveau Projet Créé : ' . $project->name;
                    $body = ""; // Removed redundant information, as it's now handled by the Blade template
                    
                    Mail::to(env('MAIL_TO_ADDRESS', config('mail.from.address')))->send(new EventTriggeredMail(
                        $subject,
                        $body,
                        $project->name,
                        $request->user()->name,
                        $ownerName,
                        $ownerOffice,
                        $serviceName,
                        $project->user_id,
                        $request->user()->profile->ICE ?? 'N/A'
                    ));
                }
            });

            return new ProjectResource($project->load('service'));

        } catch (\Throwable $e) {
            // If a file was stored before the transaction failed, delete it
            if ($path) {
                Storage::disk('public')->delete($path);
            }

            // Log the error and return a server error response
            Log::error('Project creation failed: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
            ]);

            return response()->json(['error' => 'Project creation failed. ' . $e->getMessage()], 500);
        }
    }

    /**
     * Display the specified project.
     */
    public function show(Request $request, Project $project)
    {
        $this->authorize('view', $project);
        return new ProjectResource($project->load('service'));
    }

    /**
     * Update the specified project in storage.
     */
    public function update(Request $request, Project $project)
    {
        $this->authorize('update', $project);

        $validatedData = $request->validate([
            'service_id' => 'sometimes|required|exists:services,id',
            'name' => 'sometimes|required|string|max:255',
            'description' => 'nullable|string',
            'status' => 'sometimes|required|string|max:255',
            'details' => 'nullable|array',
        ]);

        // For updates, merge with existing details if needed
        if (isset($validatedData['details'])) {
            $existingDetails = $project->details ?? [];
            $validatedData['details'] = array_merge($existingDetails, $validatedData['details']);
        }

        $project->update($validatedData);

        // After updating, broadcast the event to the user, but not on initial creation
        if ($project->wasChanged('status')) {
             event(new ProjectStatusUpdated($project));
        }

        return new ProjectResource($project->load('service'));
    }

    /**
     * Remove the specified project from storage.
     */
public function attachFile(Request $request, Project $project)
{
    Log::debug('AttachFile method called.', ['project_id' => $project->id, 'user_id' => $request->user()->id]);
    Log::debug('Request files:', $request->allFiles());
    Log::debug('Request all input:', $request->all());
    Log::debug('AttachFile: Before authorization check.');

    $this->authorize('update', $project);

    try {
        Log::debug('AttachFile: After authorization check, before validation.');
        $validatedData = $request->validate([
            'file' => 'required|file|max:102400', // 100MB max
            'file_type' => 'required|string|max:255',
        ]);
        Log::debug('AttachFile: After successful validation.', ['validated_data' => $validatedData]);
    } catch (\Illuminate\Validation\ValidationException $e) {
        Log::error('File upload validation failed for project ' . $project->id, [
            'errors' => $e->errors(),
            'request_data' => $request->all(),
            'user_id' => $request->user()->id,
        ]);
        throw $e;
    }

    $uploadedFile = $request->file('file');
    $fileType = $validatedData['file_type'];

    $directory = "uploads/{$project->user_id}/projects/{$project->id}";
    Log::debug('AttachFile: Before storing file.', ['directory' => $directory]);
    $path = $uploadedFile->store($directory, 'public');
    Log::debug('AttachFile: After storing file.', ['path' => $path]);
    $url = Storage::url($path);

    $fileMetadata = [
        'path' => $path,
        'original_name' => $uploadedFile->getClientOriginalName(),
        'uploaded_at' => now()->toDateTimeString(),
        'url' => $url,
        'file_type' => $fileType, // Include file_type in metadata
    ];

    // Return file metadata without updating the project
    Log::debug('AttachFile: Returning success response.', ['file_metadata' => $fileMetadata]);
    return response()->json($fileMetadata, 201);
}

    public function destroy(Project $project)
    {
        $this->authorize('delete', $project);
        
        // THE IMPROVEMENT: Delete the entire project folder when the project is deleted.
        $directory = "uploads/{$project->user_id}/projects/{$project->id}";
        Storage::disk('public')->deleteDirectory($directory);

        $project->delete();

        return response()->json(null, 204);
    }

    /**
     * Download a file from storage/app/public.
     */
    public function downloadFile(Request $request)
    {
        Log::info('Attempting to download file.', ['user_id' => $request->user()->id, 'is_admin' => $request->user()->tokenCan('admin'), 'requested_path' => $request->query('path')]);

        $path = $request->query('path');

        if (empty($path)) {
            Log::error('File download attempt with empty path.', ['user_id' => $request->user()->id]);
            abort(400, 'File path is required.');
        }

        // Extract user_id and project_id from the path for authorization
        // Assuming the path format is "uploads/{user_id}/projects/{project_id}/..."
        preg_match('/uploads\/(\d+)\/projects\/(\d+)\//', $path, $matches);

        if (!isset($matches[1]) || !isset($matches[2])) {
            Log::error('User ID or Project ID could not be extracted from path for download.', ['user_id' => $request->user()->id, 'requested_path' => $path]);
            abort(400, 'Invalid file path for authorization.');
        }

        $extractedUserId = (int) $matches[1];
        $projectId = (int) $matches[2];

        // Ensure the extracted user_id matches the authenticated user's ID
        if ($extractedUserId !== $request->user()->id && !$request->user()->tokenCan('admin')) {
            Log::warning('Unauthorized file access attempt: User ID mismatch.', [
                'authenticated_user_id' => $request->user()->id,
                'extracted_user_id' => $extractedUserId,
                'project_id' => $projectId,
                'requested_path' => $path
            ]);
            abort(403, 'Unauthorized access to file.');
        }

        $project = Project::findOrFail($projectId);

        // AUTHORIZATION: Use the policy to ensure the user owns this project.
        $this->authorize('view', $project);

        if (empty($path)) {
            Log::error('File download attempt with empty path.', ['user_id' => $request->user()->id]);
            abort(400, 'File path is required.');
        }

        if (!Storage::disk('public')->exists($path)) {
            Log::error('File not found for download.', ['user_id' => $request->user()->id, 'path' => $path]);
            abort(404, 'File not found.');
        }

        // Log successful file existence check
        Log::info('File found, proceeding with download.', ['user_id' => $request->user()->id, 'path' => $path]);
        
        // Return the file for download
        return Storage::disk('public')->download($path);
    }
}