<?php
namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Resources\PushTokenResource;
use App\Models\PushToken; // Ensure this model exists and points to the 'push_tokens' table
use Illuminate\Http\Request;

class PushTokenController extends Controller
{
    /**
     * Store (register) a new push notification token for the authenticated user.
     * Corresponds to POST /push-tokens
     */
    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'token' => 'required|string|max:255',
        ]);

        $user = $request->user();

        // Check if the exact token already exists for this user
        $pushToken = PushToken::firstOrCreate(
            ['user_id' => $user->id, 'token' => $validatedData['token']],
            ['user_id' => $user->id, 'token' => $validatedData['token']]
        );

        return response()->json([
            'message' => 'Push token registered successfully.',
            'token' => new PushTokenResource($pushToken),
        ], 200); // 200 OK is fine since it's an "update-or-create" operation.
    }

    /**
     * Delete (de-register) a push notification token for the authenticated user.
     * Corresponds to DELETE /push-tokens
     */
    public function destroy(Request $request)
    {
        $validatedData = $request->validate([
            'token' => 'required|string|exists:push_tokens,token', // Ensure the token exists before trying to delete
        ]);

        $user = $request->user();

        // Find and delete the specific token for the currently authenticated user.
        // It's important to delete only the token belonging to the authenticated user for security.
        $deletedCount = PushToken::where('user_id', $user->id)
                                 ->where('token', $validatedData['token'])
                                 ->delete();

        if ($deletedCount === 0) {
            // This can happen if a user tries to delete a token that belongs to someone else
            // or a token that doesn't exist. We return a success response anyway
            // to avoid revealing information.
            return response()->json(['message' => 'Push token de-registered.'], 200);
        }

        return response()->json(['message' => 'Push token de-registered successfully.'], 200);
    }
}