<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Resources\ServiceResource;
use App\Models\Service;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class ServiceController extends Controller
{
    /**
     * Display a paginated listing of services.
     * Publicly accessible.
     */
    public function index(Request $request)
    {
        $query = Service::query()->select('id', 'name', 'description', 'icon_name', 'screen_name');

        // Search functionality
        if ($request->filled('search')) {
            $searchTerm = $request->input('search');
            $query->where(function ($q) use ($searchTerm) {
                $q->where('name', 'like', "%{$searchTerm}%")
                  ->orWhere('description', 'like', "%{$searchTerm}%");
            });
        }

        $services = $query->paginate(15);

        return ServiceResource::collection($services);
    }

    /**
     * Store a newly created service in storage.
     * (Admin Only - as defined in api.php)
     */
    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'name' => 'required|string|max:255|unique:services,name',
            'description' => 'nullable|string',
            'icon_name' => 'nullable|string|max:255',
            'related_table_name' => 'nullable|string|max:255',
            'screen_name' => 'nullable|string|max:255',
            'schema' => 'nullable|array',
        ]);

        $service = Service::create($validatedData);

        return new ServiceResource($service);
    }

    /**
     * Display the specified service.
     * Publicly accessible.
     */
    public function show(Service $service)
    {
        return new ServiceResource($service);
    }

    /**
     * Update the specified service in storage.
     * (Admin Only)
     */
    public function update(Request $request, Service $service)
    {
        $validatedData = $request->validate([
            'name' => ['sometimes', 'required', 'string', 'max:255', Rule::unique('services')->ignore($service->id)],
            'description' => 'nullable|string',
            'icon_name' => 'nullable|string|max:255',
            'related_table_name' => 'nullable|string|max:255',
            'screen_name' => 'nullable|string|max:255',
            'schema' => 'nullable|array',
        ]);

        $service->update($validatedData);

        return new ServiceResource($service);
    }

    /**
     * Remove the specified service from storage.
     * (Admin Only)
     */
    public function destroy(Service $service)
    {
        $service->delete();

        return response()->json(null, 204);
    }
}
