<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Resources\UserResource; // <-- Import the new resource
use App\Events\UserDeleted;
use App\Models\User;
use App\Models\AuthCode;
use App\Notifications\SendPushNotification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Carbon\Carbon;

class UserController extends Controller
{
    /**
     * Display a paginated list of users.
     * (Admin Only - as defined in api.php)
     */
    public function index(Request $request)
    {
        $query = User::with('profile');

        // Search functionality
        if ($request->filled('search')) {
            $searchTerm = $request->input('search');
            $query->where(function ($q) use ($searchTerm) {
                $q->where('name', 'like', "%{$searchTerm}%")
                  ->orWhere('email', 'like', "%{$searchTerm}%")
                  ->orWhereHas('profile', function ($profileQuery) use ($searchTerm) {
                      $profileQuery->where('company', 'like', "%{$searchTerm}%")
                                   ->orWhere('phone_number', 'like', "%{$searchTerm}%");
                  });
            });
        }

        return UserResource::collection($query->paginate(10));
    }

    /**
     * Store a newly created user and their profile.
     * (Admin Only)
     */
    public function store(Request $request)
    {
        // Standardize on 'company' to match the database column.
        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8',
            'company' => 'nullable|string|max:255',
            'phone_number' => 'nullable|string|max:255',
            'ICE' => 'nullable|string|max:255',
            'locale' => 'sometimes|required|string|in:en,fr',
        ]);

        try {
            $user = DB::transaction(function () use ($validatedData) {
                $user = User::create([
                    'name' => $validatedData['name'],
                    'email' => $validatedData['email'],
                    'password' => Hash::make($validatedData['password']),
                ]);

                // Use the relationship to create the associated profile.
                $user->profile()->create([
                    'full_name' => $validatedData['name'],
                    'company' => $validatedData['company'] ?? null,
                    'phone_number' => $validatedData['phone_number'] ?? null,
                    'ICE' => $validatedData['ICE'] ?? null,
                    'locale' => $validatedData['locale'] ?? 'en',
                ]);

                return $user;
            });

            // Return the created user formatted by our resource.
            return new UserResource($user);
        } catch (\Throwable $e) {
            \Log::error('User creation failed: ' . $e->getMessage());
            return response()->json(['error' => 'An internal error occurred during user creation.'], 500);
        }
    }
    
    /**
     * Display the specified user.
     * (Admin Only)
     */
    public function show(User $user)
    {
        // Load the profile relationship and return the formatted resource.
        return new UserResource($user->load('profile'));
    }


    /**
     * Update the specified user's information.
     * (Admin Only)
     */
    public function update(Request $request, User $user)
    {
        // Standardize on 'company' to match the database.
        $validatedData = $request->validate([
            'name' => 'sometimes|required|string|max:255',
            'email' => 'sometimes|required|email|unique:users,email,' . $user->id,
            'company' => 'nullable|string|max:255',
            'phone_number' => 'nullable|string|max:255',
            'ICE' => 'nullable|string|max:255',
            'locale' => 'sometimes|required|string|in:en,fr',
        ]);

        try {
            DB::transaction(function () use ($validatedData, $user) {
                $user->update([
                    'name' => $validatedData['name'] ?? $user->name,
                    'email' => $validatedData['email'] ?? $user->email,
                ]);

                // Use updateOrCreate for cleaner logic. It will update the profile
                // if it exists, or create it if it doesn't.
                $user->profile()->updateOrCreate(
                    ['user_id' => $user->id], // Condition to find the profile
                    [ // Data to update or create with
                        'full_name' => $validatedData['name'] ?? $user->profile->full_name ?? $user->name,
                        'company' => $validatedData['company'] ?? $user->profile->company ?? null,
                        'ICE' => $validatedData['ICE'] ?? $user->profile->ICE ?? null,
                        'phone_number' => $validatedData['phone_number'] ?? $user->profile->phone_number ?? null,
                        'locale' => $validatedData['locale'] ?? $user->profile->locale ?? 'en',
                    ]
                );
            });

            return response()->json(['message' => "User {$user->email} updated successfully"]);
        } catch (\Throwable $e) {
            \Log::error('User update failed: ' . $e->getMessage());
            return response()->json(['error' => 'An internal error occurred during user update.'], 500);
        }
    }

    /**
     * Remove the specified user from storage.
     * (Admin Only)
     */
    public function destroy(User $user)
    {
        try {
            DB::transaction(function () use ($user) {
                // Dispatch the event before deleting the user
                event(new UserDeleted($user->id));
                
                // ON DELETE CASCADE in the DB handles the profile. This is perfect.
                $user->delete();
            });

            return response()->json(['message' => "User with ID {$user->id} deleted successfully"]);
        } catch (\Exception $e) {
            \Log::error('Error deleting user: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to delete user.'], 500);
        }
    }

    /**
     * Generate a temporary auth code for a user.
     * (Admin Only)
     */
    public function generateAuthCode(Request $request)
    {
        $validatedData = $request->validate([ 'userId' => 'required|exists:users,id' ]);

        try {
            $code = strtoupper(Str::random(8));
            $authCode = AuthCode::create([
                'user_id' => $validatedData['userId'],
                'code' => $code,
                'expires_at' => Carbon::now()->addMinutes(15),
            ]);
            return response()->json($authCode, 201);
        } catch (\Exception $e) {
            \Log::error('Failed to generate auth code: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to generate auth code.'], 500);
        }
    }

    /**
     * Display dashboard data for the authenticated client.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getClientDashboardData(Request $request)
    {
        // SECURE: Get the user from the authenticated request, not a URL parameter.
        $user = $request->user();

        // ARCHITECTURALLY CORRECT: Interact with models directly, not other controllers.
        // We load the relationships for the authenticated user.
        // Let's fetch the 5 most recent of each for a clean dashboard view.
        $projects = $user->projects()->latest()->take(5)->get();
        $invoices = $user->invoices()->latest()->take(5)->get();
        $notifications = $user->notifications()->latest()->take(5)->get();

        return response()->json([
            'projects' => $projects,
            'invoices' => $invoices,
            'notifications' => $notifications,
        ]);
    }

    /**
     * Revoke all of a user's access tokens and send a notification.
     * (Admin Only)
     */
    public function revokeSession(User $user)
    {
        // Revoke all of the user's tokens
        $user->tokens()->update(['revoked_at' => now()]);

        // Send a silent push notification to the user to trigger a client-side logout
        $user->notify(new SendPushNotification(
            'Session Revoked',
            'Your session has been revoked by an administrator.',
            'session_revoked'
        ));

        return response()->json(['message' => "All sessions for user {$user->email} have been revoked."]);
    }
}