<?php

namespace App\Models;

use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;

class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable;


    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
    ];

    //======================================================================
    // RELATIONSHIPS
    //======================================================================

    /**
     * Get the profile associated with the user.
     */
    public function profile()
    {
        return $this->hasOne(Profile::class);
    }

    /**
     * Get the user's preferred locale.
     *
     * @return string
     */
    public function getLocaleAttribute()
    {
        return $this->profile->locale ?? config('app.locale');
    }

    /**
     * Get the projects for the user.
     */
    public function projects()
    {
        return $this->hasMany(Project::class);
    }

    /**
     * Get the invoices for the user.
     */
    public function invoices()
    {
        return $this->hasMany(Invoice::class);
    }

    /**
     * The notifications that belong to the user.
     */
    public function notifications(): BelongsToMany
    {
        return $this->belongsToMany(Notification::class, 'notification_user')
                    ->withPivot('read_at')
                    ->withTimestamps()
                    ->orderByPivot('created_at', 'desc'); // Order by when it was received
    }

    /**
     * Get the auth codes for the user.
     * @added
     */
    public function authCodes()
    {
        return $this->hasMany(AuthCode::class);
    }

    /**
     * Get the push notification tokens for the user.
     * @added
     */
    public function pushTokens()
    {
        // Assuming your model is named PushToken. Adjust if necessary.
        return $this->hasMany(PushToken::class);
    }

    /**
     * Route notifications for the Firebase Cloud Messaging channel.
     *
     * @param  \Illuminate\Notifications\Notification  $notification
     * @return array<string>
     */
    public function routeNotificationForFcm($notification): array
    {
        // Ensure only unique tokens are returned to prevent duplicate notifications
        return $this->pushTokens->pluck('token')->unique()->toArray();
    }

    /**
     * Check if the user is an admin.
     *
     * @return bool
     */
    public function isAdmin()
    {
        return false;
    }
}