<?php

namespace App\Notifications;

use Illuminate\Notifications\Notification;
use NotificationChannels\Fcm\FcmChannel;
use NotificationChannels\Fcm\FcmMessage;
use NotificationChannels\Fcm\Resources\AndroidConfig;
use NotificationChannels\Fcm\Resources\AndroidMessagePriority;
use NotificationChannels\Fcm\Resources\AndroidNotification;
use NotificationChannels\Fcm\Resources\ApnsConfig;
use NotificationChannels\Fcm\Resources\ApnsFcmOptions;
use NotificationChannels\Fcm\Resources\Notification as FcmNotification;
use Illuminate\Support\Facades\Log; // Import the Log facade
 
class SendPushNotification extends Notification
{
    // Removed Queueable trait for synchronous execution
 
    protected $title;
    protected $body;
    protected $notificationType;
    protected $data;

    /**
     * Create a new notification instance.
     */
    public function __construct(string $title, string $body, string $notificationType = 'user', array $data = [])
    {
        $this->title = $title;
        $this->body = $body;
        $this->notificationType = $notificationType;
        $this->data = $data;
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via($notifiable): array
    {
        Log::info('SendPushNotification: via method called.', [
            'notifiable_id' => $notifiable->id,
            'notifiable_type' => get_class($notifiable),
        ]);
        return [FcmChannel::class];
    }

    /**
     * Get the FCM representation of the notification.
     */
    public function toFcm($notifiable): FcmMessage
    {
        Log::info('SendPushNotification: toFcm method called.', [
            'notifiable_id' => $notifiable->id,
            'notifiable_type' => get_class($notifiable),
            'notification_title' => $this->title,
            'notification_body' => $this->body,
            'notification_type' => $this->notificationType,
            'notification_data_payload' => $this->data,
        ]);
        $fcmMessage = FcmMessage::create();

        // Base data payload
        $baseData = array_merge($this->data, [
            'title' => $this->title,
            'body' => $this->body,
            'type' => $this->notificationType,
            'timestamp' => now()->toISOString(),
        ]);

        // If it's a session revocation, send a silent, data-only notification
        if ($this->notificationType === 'session_revoked') {
            return $fcmMessage
                ->setData($baseData) // Only data
                ->setAndroid(
                    AndroidConfig::create()->setPriority(AndroidMessagePriority::HIGH())
                )
                ->setApns(
                    ApnsConfig::create()->setPayload([
                        'aps' => [
                            'content-available' => 1, // Key for silent notifications on iOS
                        ],
                    ])
                );
        }

        // For all other notifications, send a standard visible notification
        return $fcmMessage
            ->setNotification(
                FcmNotification::create()
                    ->setTitle($this->title)
                    ->setBody($this->body)
            )
            ->setData(array_merge($baseData, [
                 'click_action' => 'FLUTTER_NOTIFICATION_CLICK',
            ]))
            ->setApns(
                ApnsConfig::create()
                    ->setFcmOptions(
                        ApnsFcmOptions::create()->setImage(null)
                    )
                    ->setPayload([
                        'aps' => [
                            'alert' => [
                                'title' => $this->title,
                                'body' => $this->body,
                            ],
                            'sound' => 'default',
                            'badge' => 1,
                        ],
                    ])
            )
            ->setAndroid(
                AndroidConfig::create()
                    ->setPriority(AndroidMessagePriority::HIGH())
                    ->setNotification(
                        AndroidNotification::create()
                            ->setTitle($this->title)
                            ->setBody($this->body)
                            ->setSound('default')
                            ->setClickAction('FLUTTER_NOTIFICATION_CLICK')
                    )
            );
    }

    /**
     * Get the FCM device tokens for the notifiable.
     */
    public function routeNotificationForFcm($notifiable)
    {
        $tokens = $notifiable->pushTokens()->pluck('token')->unique()->toArray();
        Log::info('SendPushNotification: routeNotificationForFcm method called.', [
            'notifiable_id' => $notifiable->id,
            'notifiable_type' => get_class($notifiable),
            'retrieved_push_tokens' => $tokens,
        ]);
        return $tokens;
    }
}