<?php

namespace App\Observers;

use App\Models\Notification as AppNotification;
use App\Models\User;
use App\Notifications\SendPushNotification;
use Illuminate\Support\Facades\Log;

class NotificationObserver
{
    /**
     * Handle the Notification "created" event.
     * This method will be called AUTOMATICALLY every time a new notification
     * is saved to the database.
     *
     * @param  \App\Models\Notification  $notification
     * @return void
     */
    public function created(AppNotification $notification)
    {
        Log::info('NotificationObserver: Notification created, sending push notifications directly.', [
            'notification_id' => $notification->id,
            'type' => $notification->type,
            'notification_data' => $notification->toArray(), // Add full notification data
            'attached_users_count_before_check' => $notification->users()->count(), // Log attached users count
        ]);

        $users = collect();
        
        if ($notification->type === 'broadcast') {
            $users = User::whereHas('pushTokens')->get();
            Log::info('Observer: Broadcasting to all users with push tokens.', [
                'notification_id' => $notification->id,
                'users_count' => $users->count()
            ]);
        } else {
            // Get all attached users
            // Retrieve users directly associated with the notification via the pivot table
            $users = $notification->users()->whereHas('pushTokens')->get();
            Log::info('Observer: Sending to specific users.', [
                'notification_id' => $notification->id,
                'users_count' => $users->count(),
                'users_ids' => $users->pluck('id')->toArray(), // Log IDs of users found
            ]);
        }

        if ($users->isEmpty()) {
            Log::info('Observer: No users with push tokens found for this notification.', [
                'notification_id' => $notification->id
            ]);
            return;
        }

        foreach ($users as $user) {
            try {
                $deviceTokens = $user->pushTokens()->pluck('push_token')->toArray();
                Log::info('Observer: Retrieved device tokens for user.', [
                    'notification_id' => $notification->id,
                    'user_id' => $user->id,
                    'device_tokens' => $deviceTokens, // Log the actual device tokens
                ]);

                if (!empty($deviceTokens)) {
                    // Send the push notification
                    $user->notify(new SendPushNotification(
                        $notification->title,
                        $notification->body,
                        $notification->type
                    ));
                    
                    Log::info('Observer: Push notification sent successfully.', [
                        'notification_id' => $notification->id,
                        'user_id' => $user->id,
                        'device_tokens_count' => count($deviceTokens),
                        'type' => $notification->type
                    ]);
                } else {
                    Log::info('Observer: No device tokens found for user.', [
                        'notification_id' => $notification->id,
                        'user_id' => $user->id
                    ]);
                }
            } catch (\Exception $e) {
                Log::error('Observer: Error sending push notification.', [
                    'notification_id' => $notification->id,
                    'user_id' => $user->id,
                    'error' => $e->getMessage()
                ]);
            }
        }
    }
}