<?php

namespace App\Policies;

use App\Models\Notification;
use App\Models\User;
use Illuminate\Auth\Access\HandlesAuthorization;

class NotificationPolicy
{
    use HandlesAuthorization;

    /**
     * Grant all permissions to admin users before other checks.
     */
    public function before($user, $ability) // Removed User type hint to allow Admin model
    {
        // Your existing admin check is perfect.
        if ($user->tokenCan('admin')) {
            return true;
        }
    }

    /**
     * Determine whether the user can view the list of notifications.
     * The controller already scopes this, so we just need to allow access.
     */
    public function viewAny(User $user)
    {
        return true; // Any authenticated user can view their own notification list.
    }

    /**
     * Determine whether the user can view a specific notification.
     */
    public function view(User $user, Notification $notification)
    {
        // A user can view any broadcast notification.
        if ($notification->type === 'broadcast') {
            return true;
        }

        // Or they can view a notification if they are a recipient.
        return $notification->users()->where('user_id', $user->id)->exists();
    }

    /**
     * Determine whether the user can create models.
     */
    public function create(User $user)
    {
        // Your existing create logic is perfect.
        return $user->tokenCan('admin');
    }

    /**
     * Determine whether the user can delete the model (i.e., hide it).
     * This is the missing method that will fix your problem.
     */
    public function delete(User $user, Notification $notification)
    {
        // A user can "delete" (hide) any broadcast notification.
        if ($notification->type === 'broadcast') {
            return true;
        }

        // Or they can delete a notification if they are a recipient.
        return $notification->users()->where('user_id', $user->id)->exists();
    }
}