<?php

namespace App\Services;

use App\Models\User;
use App\Models\Notification;
use Illuminate\Support\Facades\Log; // Import the Log facade
use Illuminate\Support\Facades\Lang; // Import the Lang facade
use App\Notifications\SendPushNotification; // Import SendPushNotification

class NotificationService
{
    /**
     * Creates a database notification and dispatches a job to send push notifications.
     *
     * @param User $user The user who will receive the notification.
     * @param string $titleKey The translation key for the title of the notification.
     * @param string $bodyKey The translation key for the main content of the notification.
     * @param string $type The type of notification (e.g., "project_status_updated", "invoice_created").
     * @param array $data Optional additional data for translation placeholders.
     * @return Notification The created Notification model instance.
     */
    public static function create(User $user, string $titleKey, string $bodyKey, string $type, array $data = []): Notification
    {
        $locale = $user->profile->locale ?? config('app.locale'); // Get user's locale or fallback to app locale

        $translatedTitle = Lang::get($titleKey, $data, $locale);
        $translatedBody = Lang::get($bodyKey, $data, $locale);

        $notification = Notification::create([
            'title'    => $translatedTitle,
            'body'     => $translatedBody,
            'type'     => $type,
            'data'     => $data, // Store the data payload
        ]);

        $user->notifications()->attach($notification->id);
        Log::info('NotificationService: User attached to notification.', [
            'notification_id' => $notification->id,
            'attached_user_id' => $user->id,
            'current_attached_users_count' => $notification->users()->count(), // Check count after attach
        ]);
        $notification->load('users'); // Reload the notification with its attached users
        self::sendPushNotifications($notification); // Send push notifications

        return $notification;
    }

    /**
     * Sends push notifications for a given Notification model.
     * This method encapsulates the logic previously found in the NotificationObserver.
     *
     * @param \App\Models\Notification $notification The notification instance.
     * @return void
     */
    public static function sendPushNotifications(\App\Models\Notification $notification): void
    {
        Log::info('NotificationService: sendPushNotifications called.', [
            'notification_id' => $notification->id,
            'type' => $notification->type,
            'notification_data' => $notification->toArray(),
        ]);

        $users = collect();

        if ($notification->type === 'broadcast') {
            $users = User::whereHas('pushTokens')->get();
            Log::info('NotificationService: Broadcasting to all users with push tokens.', [
                'notification_id' => $notification->id,
                'users_count' => $users->count()
            ]);
        } else {
            // Retrieve users directly associated with the notification via the pivot table
            // Ensure users and their pushTokens are loaded
            $notification->load('users.pushTokens');
            $users = $notification->users;
            Log::info('NotificationService: Sending to specific users.', [
                'notification_id' => $notification->id,
                'users_count' => $users->count(),
                'users_ids' => $users->pluck('id')->toArray(),
            ]);
        }

        if ($users->isEmpty()) {
            Log::info('NotificationService: No users with push tokens found for this notification.', [
                'notification_id' => $notification->id
            ]);
            return;
        }

        foreach ($users as $user) {
            try {
                $deviceTokens = $user->pushTokens->pluck('token')->toArray();
                Log::info('NotificationService: Retrieved device tokens for user.', [
                    'notification_id' => $notification->id,
                    'user_id' => $user->id,
                    'device_tokens' => $deviceTokens,
                ]);

                if (!empty($deviceTokens)) {
                    $user->notify(new SendPushNotification(
                        $notification->title, // Already translated and stored in DB
                        $notification->body,  // Already translated and stored in DB
                        $notification->type,
                        $notification->data ?? [] // Pass the data payload
                    ));

                    Log::info('NotificationService: Push notification sent successfully.', [
                        'notification_id' => $notification->id,
                        'user_id' => $user->id,
                        'device_tokens_count' => count($deviceTokens),
                        'type' => $notification->type
                    ]);
                } else {
                    Log::info('NotificationService: No device tokens found for user.', [
                        'notification_id' => $notification->id,
                        'user_id' => $user->id
                    ]);
                }
            } catch (\Exception $e) {
                Log::error('NotificationService: Error sending push notification.', [
                    'notification_id' => $notification->id,
                    'user_id' => $user->id,
                    'error' => $e->getMessage()
                ]);
            }
        }
    }
}