<?php


use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Api\AdminController;
use App\Http\Controllers\Api\AdvertisementController;
use App\Http\Controllers\Api\AuthController;
use App\Http\Controllers\Api\InvoiceController;
use App\Http\Controllers\Api\LogController; // Corrected Naming
use App\Http\Controllers\Api\NotificationController;
use App\Http\Controllers\Api\ProfileController;
use App\Http\Controllers\Api\ProjectController;
use App\Http\Controllers\Api\PushTokenController;
use App\Http\Controllers\Api\ServiceController;
use App\Http\Controllers\Api\UserController;
use App\Http\Controllers\Api\AuthCodeController;
use App\Http\Controllers\Api\MailController;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Refactored for clarity, security, and consistency.
|
*/


//==========================================================================
// Public Routes (No Authentication Required)
//==========================================================================
Route::post('/login', [AuthController::class, 'login']);
Route::post('/validate-code', [AuthController::class, 'validateCode']);

// Email Sending Route
Route::post('/send-email', [MailController::class, 'send']);


//==========================================================================
// Authenticated User Routes
//==========================================================================
Route::middleware('auth:sanctum')->group(function () {
    // --- Authentication & Profile ---
    Route::post('/logout', [AuthController::class, 'logout']);
    Route::get('/user', function (Request $request) {
        $user = $request->user();
        if ($user instanceof \App\Models\Admin) {
            return response()->json([
                'id' => $user->id,
                'name' => $user->username,
                'role' => 'admin',
            ]);
        } elseif ($user instanceof \App\Models\User) {
            return response()->json([
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'role' => 'user',
                'locale' => $user->profile->locale ?? config('app.locale'),
            ]);
        }
        return response()->json(null, 401); // Unauthorized if no user found
    });

    // --- Profile Management (Scoped to the authenticated user) ---
    Route::get('/my-profile', [ProfileController::class, 'show']);
    Route::put('/my-profile', [ProfileController::class, 'update']);

    // --- Core Features (Scoped to the authenticated user) ---
    Route::apiResource('projects', ProjectController::class);
    // Add this new route for uploading files specifically to a project
    Route::post('/projects/{project}/files', [ProjectController::class, 'attachFile'])
        ->middleware('auth:api');
    Route::get('notifications', [NotificationController::class, 'index']);
    Route::post('notifications/{notification}/read', [NotificationController::class, 'markAsRead']);
    Route::delete('notifications/{notification}', [NotificationController::class, 'destroy']);
    Route::get('notifications/unread-count', [NotificationController::class, 'unreadCount']);
    Route::post('notifications/{notification}/restore', [NotificationController::class, 'restore']); // Optional restore functionality
    Route::get('/my-invoices', [InvoiceController::class, 'index']);
    Route::get('/my-invoices/{invoice}', [InvoiceController::class, 'show']);
    Route::get('/my-invoices/{invoice}/download', [InvoiceController::class, 'download'])->name('invoices.download');
    Route::get('/client-dashboard', [UserController::class, 'getClientDashboardData']); // SECURE: No {userId}, gets data for the logged-in user

    // --- Publicly Viewable Resources ---
    Route::get('/services', [ServiceController::class, 'index']); // Any user can see available services
    Route::get('/services/{service}', [ServiceController::class, 'show']);
    Route::get('/advertisements', [AdvertisementController::class, 'index']);


    // --- Push Notifications ---
    // Change 'register' to 'store' and 'delete' to 'destroy'
    Route::post('/push-tokens', [PushTokenController::class, 'store']);
    Route::delete('/push-tokens', [PushTokenController::class, 'destroy']);
});


//==========================================================================
// Admin Routes
//==========================================================================
Route::middleware(['auth:sanctum', 'admin'])->prefix('admin')->group(function () {
    // --- Full Resource Management ---
    // All actions for these resources are defined here in one place.
    Route::apiResource('users', UserController::class);
    Route::apiResource('services', ServiceController::class);
    Route::apiResource('invoices', InvoiceController::class);
    Route::get('/invoices/{invoice}/download', [InvoiceController::class, 'download'])->name('admin.invoices.download');
    Route::apiResource('admins', AdminController::class);
    Route::put('admins/{id}/reset-password', [AdminController::class, 'resetPassword']);
    Route::apiResource('advertisements', AdvertisementController::class);
    Route::post('advertisements/{advertisement}/toggle-active', [AdvertisementController::class, 'toggleActive']);
    Route::apiResource('projects', ProjectController::class)->only(['index', 'show', 'destroy']); // Admins can view/delete any project
    Route::put('projects/{project}', [ProjectController::class, 'update']); // Allow admins to update projects

    // --- Admin-Specific Actions ---
    Route::post('/users/generate-code', [UserController::class, 'generateAuthCode']);
    Route::post('/users/{user}/revoke-session', [UserController::class, 'revokeSession']);
    Route::post('/users/{user}/send-revoke-session-notification', [AdminController::class, 'sendRevokeSessionNotification']); // New route
    Route::post('notifications', [NotificationController::class, 'store']);

    // --- Auth Code Management (for viewing/deleting user codes) ---
    Route::apiResource('auth-codes', AuthCodeController::class)->only(['index', 'destroy']);
    Route::get('/logs', [LogController::class, 'index']);
    Route::get('/download-file', [ProjectController::class, 'downloadFile']); // New route for downloading files

    
});

Route::get('/test-notification', function() {
    $user = auth()->user();
    
    // Create a test notification
    $user->notifications()->create([
        'category' => 'Test',
        'message' => 'This is a test notification',
    ]);
    
    return 'Test notification created';
});