<?php

namespace NotificationChannels\Fcm;

use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Contracts\Events\Dispatcher;
use Illuminate\Notifications\Events\NotificationFailed;
use Illuminate\Notifications\Notification;
use Kreait\Firebase\Exception\MessagingException;
use Kreait\Firebase\Messaging\CloudMessage;
use Kreait\Firebase\Messaging\Message;
use NotificationChannels\Fcm\Exceptions\CouldNotSendNotification;
use ReflectionException;
use Throwable;

class FcmChannel
{
    const MAX_TOKEN_PER_REQUEST = 500;

    /**
     * @var \Illuminate\Contracts\Events\Dispatcher
     */
    protected $events;

    /**
     * FcmChannel constructor.
     *
     * @param  \Illuminate\Contracts\Events\Dispatcher  $dispatcher
     */
    public function __construct(Dispatcher $dispatcher)
    {
        $this->events = $dispatcher;
    }

    /**
     * @var string|null
     */
    protected $fcmProject = null;

    /**
     * Send the given notification.
     *
     * @param  mixed  $notifiable
     * @param  \Illuminate\Notifications\Notification  $notification
     * @return array
     *
     * @throws \NotificationChannels\Fcm\Exceptions\CouldNotSendNotification
     * @throws \Kreait\Firebase\Exception\FirebaseException
     */
    public function send($notifiable, Notification $notification)
    {
        $token = $notifiable->routeNotificationFor('fcm', $notification);

        if (empty($token)) {
            return [];
        }

        // Get the message from the notification class
        $fcmMessage = $notification->toFcm($notifiable);

        if (! $fcmMessage instanceof Message) {
            throw CouldNotSendNotification::invalidMessage();
        }

        $this->fcmProject = null;
        if (method_exists($notification, 'fcmProject')) {
            $this->fcmProject = $notification->fcmProject($notifiable, $fcmMessage);
        }

        $responses = [];

        try {
            // If token is an array, send to each token individually so the SDK uses the project-specific v1 URL
            if (is_array($token)) {
                $partialTokens = array_chunk($token, self::MAX_TOKEN_PER_REQUEST, false);

                foreach ($partialTokens as $tokensChunk) {
                    foreach ($tokensChunk as $singleToken) {
                        try {
                            // Clone FcmMessage instances to avoid mutating the original when setToken() is used
                            $messageToSend = $fcmMessage;
                            if ($fcmMessage instanceof FcmMessage) {
                                $messageToSend = clone $fcmMessage;
                            }

                            $responses[] = $this->sendToFcm($messageToSend, $singleToken);
                        } catch (MessagingException $e) {
                            // dispatch failed event for this token and continue with others
                            $this->failedNotification($notifiable, $notification, $e, $singleToken);

                            // keep a structured error in responses so caller can inspect
                            $responses[] = [
                                'success' => false,
                                'token' => $singleToken,
                                'error' => $e->getMessage(),
                            ];
                        }
                    }
                }
            } else {
                // Single token: normal path
                $messageToSend = $fcmMessage;
                if ($fcmMessage instanceof FcmMessage) {
                    $messageToSend = clone $fcmMessage;
                }

                $responses[] = $this->sendToFcm($messageToSend, $token);
            }
        } catch (MessagingException $exception) {
            // Something unexpected at a higher level
            $this->failedNotification($notifiable, $notification, $exception, $token);
            throw CouldNotSendNotification::serviceRespondedWithAnError($exception);
        }

        return $responses;
    }

    /**
     * @return \Kreait\Firebase\Messaging
     */
    protected function messaging()
    {
        try {
            $messaging = app('firebase.manager')->project($this->fcmProject)->messaging();
        } catch (BindingResolutionException $e) {
            $messaging = app('firebase.messaging');
        } catch (ReflectionException $e) {
            $messaging = app('firebase.messaging');
        }

        return $messaging;
    }

    /**
     * @param  \Kreait\Firebase\Messaging\Message  $fcmMessage
     * @param $token
     * @return mixed
     *
     * @throws \Kreait\Firebase\Exception\MessagingException
     * @throws \Kreait\Firebase\Exception\FirebaseException
     */
    protected function sendToFcm(Message $fcmMessage, $token)
    {
        // CloudMessage::withChangedTarget returns a new instance (immutable)
        if ($fcmMessage instanceof CloudMessage) {
            $fcmMessage = $fcmMessage->withChangedTarget('token', $token);
        }

        // FcmMessage::setToken may be mutating, that's why we clone earlier in send()
        if ($fcmMessage instanceof FcmMessage) {
            $fcmMessage->setToken($token);
        }

        return $this->messaging()->send($fcmMessage);
    }

    /**
     * @param $fcmMessage
     * @param  array  $tokens
     * @return \Kreait\Firebase\Messaging\MulticastSendReport
     *
     * @throws \Kreait\Firebase\Exception\MessagingException
     * @throws \Kreait\Firebase\Exception\FirebaseException
     */
    protected function sendToFcmMulticast($fcmMessage, array $tokens)
    {
        // Left here in case you want to use SDK multicast in the future. Be aware
        // that some SDK versions use the global /batch endpoint instead of the
        // project-specific v1 path. Use with null-checks when building a MulticastMessage.
        return $this->messaging()->sendMulticast($fcmMessage, $tokens);
    }

    /**
     * Dispatch failed event.
     *
     * @param  mixed  $notifiable
     * @param  \Illuminate\Notifications\Notification  $notification
     * @param  \Throwable  $exception
     * @param  string|array  $token
     * @return array|null
     */
    protected function failedNotification($notifiable, Notification $notification, Throwable $exception, $token)
    {
        return $this->events->dispatch(new NotificationFailed(
            $notifiable,
            $notification,
            self::class,
            [
                'message' => $exception->getMessage(),
                'exception' => $exception,
                'token' => $token,
            ]
        ));
    }
}
